<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class CreateEventosSchemaAndTables extends Migration
{
    public function up(): void
    {
        // Crear esquema "eventos" si no existe (PostgreSQL)
        DB::statement('CREATE SCHEMA IF NOT EXISTS eventos');

        /*
         |--------------------------------------------------------------------------
         | TABLAS BASE / CATÁLOGOS
         |--------------------------------------------------------------------------
         */

        // 1) tipos_evento
        Schema::create('eventos.tipos_evento', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('nombre', 100);
            $table->text('descripcion')->nullable();
            $table->string('color_hex', 7)->nullable();
            $table->boolean('activo')->default(true);
        });

        // 2) sedes
        Schema::create('eventos.sedes', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('nombre', 150);
            $table->string('direccion', 255)->nullable();
            $table->string('ciudad', 100)->nullable();
            $table->boolean('activo')->default(true);
        });

        // 3) roles_participante
        Schema::create('eventos.roles_participante', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('nombre', 100);
            $table->text('descripcion')->nullable();
        });

        // 4) recursos
        Schema::create('eventos.recursos', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('nombre', 150);
            $table->string('tipo', 100)->nullable();
            $table->string('codigo_interno', 50)->nullable();
            $table->string('estado', 20)->default('operativo');
            $table->text('descripcion')->nullable();
            $table->boolean('activo')->default(true);
        });

        // 5) proveedores
        Schema::create('eventos.proveedores', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('nombre', 200);
            $table->string('tipo', 50)->nullable();      // empresa, persona, docente, etc.
            $table->string('documento', 20)->nullable(); // RUC / DNI
            $table->string('correo', 150)->nullable();
            $table->string('telefono', 50)->nullable();
            $table->string('direccion', 255)->nullable();
            $table->text('observaciones')->nullable();
            $table->boolean('activo')->default(true);
        });

        // 6) conceptos_gasto
        Schema::create('eventos.conceptos_gasto', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('nombre', 150);
            $table->string('categoria', 100)->nullable();   // Alquiler, Personal, Difusión...
            $table->string('unidad_ref', 50)->nullable();   // hora, servicio, unidad...
            $table->text('descripcion')->nullable();
            $table->boolean('activo')->default(true);
        });

        // 7) personas (docentes, estudiantes, invitados, etc.)
        Schema::create('eventos.personas', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('nombres', 150);
            $table->string('apellidos', 150);
            $table->string('tipo_persona', 30)->nullable(); // docente, estudiante, invitado, staff
            $table->string('documento', 20)->nullable();
            $table->string('correo', 150)->nullable();
            $table->string('telefono', 50)->nullable();
            $table->string('especialidad', 150)->nullable(); // instrumento, área, etc.
            $table->boolean('es_interno')->default(true);
            $table->boolean('activo')->default(true);
        });

        // 8) roles de usuario del sistema
        Schema::create('eventos.roles', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('nombre', 50);     // admin, coordinador, docente, logistica...
            $table->text('descripcion')->nullable();
        });

        // 9) usuarios del sistema
        Schema::create('eventos.usuarios', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('nombre', 150);
            $table->string('email', 150)->unique();
            $table->string('password', 255);
            $table->unsignedBigInteger('rol_id')->nullable();
            $table->unsignedBigInteger('persona_id')->nullable();
            $table->boolean('activo')->default(true);
            $table->timestamps();

            $table->foreign('rol_id')
                ->references('id')->on('eventos.roles');
            $table->foreign('persona_id')
                ->references('id')->on('eventos.personas');
        });

        /*
         |--------------------------------------------------------------------------
         | ESPACIOS FÍSICOS Y EVENTOS
         |--------------------------------------------------------------------------
         */

        // 10) espacios
        Schema::create('eventos.espacios', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->unsignedBigInteger('sede_id')->nullable();
            $table->string('nombre', 150);
            $table->string('tipo', 50)->nullable();  // Auditorio, Aula, Sala de ensayo...
            $table->integer('capacidad')->nullable();
            $table->text('descripcion')->nullable();
            $table->boolean('activo')->default(true);

            $table->foreign('sede_id')
                ->references('id')->on('eventos.sedes');
        });

        // 11) eventos (núcleo)
        Schema::create('eventos.eventos', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('titulo', 200);
            $table->text('descripcion')->nullable();
            $table->unsignedBigInteger('tipo_evento_id')->nullable();
            $table->string('modalidad', 20)->nullable();    // presencial, virtual, hibrido
            $table->unsignedBigInteger('sede_id')->nullable();
            $table->unsignedBigInteger('espacio_id')->nullable();
            $table->dateTime('fecha_inicio');
            $table->dateTime('fecha_fin');
            $table->boolean('es_todo_el_dia')->default(false);
            $table->integer('aforo_maximo')->nullable();
            $table->string('publico_objetivo', 50)->nullable(); // interno, externo, mixto
            $table->unsignedBigInteger('organizador_id')->nullable(); // luego puedes enlazar con usuarios/personas
            $table->string('estado', 20)->default('borrador'); // borrador, en_revision, aprobado...
            $table->boolean('es_academico')->default(false);
            $table->text('observaciones')->nullable();
            $table->timestamps();
            $table->softDeletes();

            $table->foreign('tipo_evento_id')
                ->references('id')->on('eventos.tipos_evento');
            $table->foreign('sede_id')
                ->references('id')->on('eventos.sedes');
            $table->foreign('espacio_id')
                ->references('id')->on('eventos.espacios');
        });

        /*
         |--------------------------------------------------------------------------
         | RELACIONES CON RECURSOS Y PARTICIPANTES
         |--------------------------------------------------------------------------
         */

        // 12) evento_recurso
        Schema::create('eventos.evento_recurso', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->unsignedBigInteger('evento_id');
            $table->unsignedBigInteger('recurso_id');
            $table->integer('cantidad')->default(1);
            $table->dateTime('uso_desde')->nullable();
            $table->dateTime('uso_hasta')->nullable();
            $table->text('observaciones')->nullable();

            $table->foreign('evento_id')
                ->references('id')->on('eventos.eventos')->onDelete('cascade');
            $table->foreign('recurso_id')
                ->references('id')->on('eventos.recursos');
        });

        // 13) evento_participantes
        Schema::create('eventos.evento_participantes', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->unsignedBigInteger('evento_id');
            $table->unsignedBigInteger('persona_id');
            $table->unsignedBigInteger('rol_participante_id')->nullable();
            $table->integer('orden_programa')->nullable();
            $table->text('descripcion_rol')->nullable();
            $table->timestamps();

            $table->foreign('evento_id')
                ->references('id')->on('eventos.eventos')->onDelete('cascade');
            $table->foreign('persona_id')
                ->references('id')->on('eventos.personas');
            $table->foreign('rol_participante_id')
                ->references('id')->on('eventos.roles_participante');
        });

        /*
         |--------------------------------------------------------------------------
         | INSCRIPCIONES Y ASISTENCIA
         |--------------------------------------------------------------------------
         */

        // 14) inscripciones
        Schema::create('eventos.inscripciones', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->unsignedBigInteger('evento_id');
            $table->unsignedBigInteger('persona_id')->nullable(); // si es interno
            $table->string('nombre', 150)->nullable();             // para externo
            $table->string('apellidos', 150)->nullable();
            $table->string('documento', 20)->nullable();
            $table->string('correo', 150)->nullable();
            $table->string('tipo_asistente', 30)->nullable();      // estudiante, docente, externo...
            $table->string('estado', 20)->default('registrado');   // registrado, confirmado, espera...
            $table->string('fuente', 20)->nullable();              // web, interno, importado...
            $table->dateTime('fecha_inscripcion')->useCurrent();
            $table->timestamps();

            $table->foreign('evento_id')
                ->references('id')->on('eventos.eventos')->onDelete('cascade');
            $table->foreign('persona_id')
                ->references('id')->on('eventos.personas');
        });

        // 15) checkins
        Schema::create('eventos.checkins', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->unsignedBigInteger('evento_id');
            $table->unsignedBigInteger('inscripcion_id')->nullable();
            $table->unsignedBigInteger('persona_id')->nullable();
            $table->dateTime('fecha_hora')->useCurrent();
            $table->string('canal', 20)->nullable();        // qr, manual, importado
            $table->text('observaciones')->nullable();

            $table->foreign('evento_id')
                ->references('id')->on('eventos.eventos')->onDelete('cascade');
            $table->foreign('inscripcion_id')
                ->references('id')->on('eventos.inscripciones');
            $table->foreign('persona_id')
                ->references('id')->on('eventos.personas');
        });

        /*
         |--------------------------------------------------------------------------
         | MÓDULO LOGÍSTICO / GASTOS
         |--------------------------------------------------------------------------
         */

        Schema::create('eventos.eventos_gastos', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->unsignedBigInteger('evento_id')->nullable();
            $table->string('tipo_registro', 15);   // PRESUPUESTO o REAL
            $table->unsignedBigInteger('concepto_id')->nullable();
            $table->string('concepto_libre', 255);
            $table->decimal('cantidad', 12, 2)->default(1);
            $table->string('unidad', 50)->nullable();        // hora, servicio, unidad...
            $table->decimal('costo_unitario', 12, 2)->default(0);
            $table->string('moneda', 10)->default('PEN');
            $table->decimal('subtotal', 12, 2)->default(0);  // lo calculas en el código
            $table->date('fecha_gasto')->nullable();
            $table->unsignedBigInteger('proveedor_id')->nullable();
            $table->string('forma_pago', 30)->nullable();         // efectivo, transferencia...
            $table->string('comprobante_tipo', 30)->nullable();   // Factura, Boleta, R/H...
            $table->string('comprobante_numero', 50)->nullable();
            $table->string('estado_pago', 20)->default('pendiente'); // pendiente, pagado, anulado
            $table->text('observaciones')->nullable();
            $table->timestamps();

            $table->foreign('evento_id')
                ->references('id')->on('eventos.eventos')->onDelete('cascade');
            $table->foreign('concepto_id')
                ->references('id')->on('eventos.conceptos_gasto');
            $table->foreign('proveedor_id')
                ->references('id')->on('eventos.proveedores');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // Borrar en orden inverso por dependencias
        Schema::dropIfExists('eventos.eventos_gastos');
        Schema::dropIfExists('eventos.checkins');
        Schema::dropIfExists('eventos.inscripciones');
        Schema::dropIfExists('eventos.evento_participantes');
        Schema::dropIfExists('eventos.evento_recurso');
        Schema::dropIfExists('eventos.eventos');
        Schema::dropIfExists('eventos.espacios');
        Schema::dropIfExists('eventos.usuarios');
        Schema::dropIfExists('eventos.roles');
        Schema::dropIfExists('eventos.personas');
        Schema::dropIfExists('eventos.conceptos_gasto');
        Schema::dropIfExists('eventos.proveedores');
        Schema::dropIfExists('eventos.recursos');
        Schema::dropIfExists('eventos.roles_participante');
        Schema::dropIfExists('eventos.sedes');
        Schema::dropIfExists('eventos.tipos_evento');

        // Opcional: si quieres borrar el esquema entero
        // DB::statement('DROP SCHEMA IF EXISTS eventos CASCADE');
    }
}
